<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Crypto Dashboard</title>
  <!-- Tailwind CSS via CDN -->
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-black text-white font-sans">
  <div class="container mx-auto p-4">
    <!-- Header -->
  

    <!-- Main Coin Section (Bitcoin) -->
    <section id="main-coin" class="bg-gray-900 rounded-lg p-6 mb-6">
      <div class="text-center text-gray-400 py-4">
        Loading Bitcoin data...
      </div>
    </section>

    <!-- Coins Sections -->
    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
      <!-- Top Cryptocurrencies -->
      <div class="bg-gray-900 rounded-lg p-6">
        <h2 class="text-lg font-bold text-green-400 mb-4">Top Cryptocurrencies</h2>
        <ul id="top-coins">
          <div class="text-center text-gray-400 py-4">
            Loading top coins...
          </div>
        </ul>
      </div>
      <!-- Recent Additions -->
      <div class="bg-gray-900 rounded-lg p-6">
        <h2 class="text-lg font-bold text-green-400 mb-4">Recent Additions</h2>
        <ul id="recent-coins">
          <div class="text-center text-gray-400 py-4">
            Loading recent coins...
          </div>
        </ul>
      </div>
    </div>
  </div>

  <script>
    // Set main coin to Bitcoin
    const mainCoinId = 'bitcoin';

    // Function to format numbers with commas
    function formatNumber(num) {
      return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
    }

    // Function to format currency
    function formatCurrency(num) {
      return new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: 'USD',
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
      }).format(num);
    }

    // Function to format large numbers (market cap, volume)
    function formatLargeNumber(num) {
      if (num >= 1e9) {
        return '$' + (num / 1e9).toFixed(2) + 'B';
      } else if (num >= 1e6) {
        return '$' + (num / 1e6).toFixed(2) + 'M';
      } else if (num >= 1e3) {
        return '$' + (num / 1e3).toFixed(2) + 'K';
      } else {
        return '$' + num.toFixed(2);
      }
    }

    // Fetch main coin data for Bitcoin
    async function fetchMainCoinData() {
      try {
        const response = await fetch(`https://api.coingecko.com/api/v3/coins/${mainCoinId}?localization=false&tickers=false&market_data=true&community_data=false&developer_data=false`);
        if (!response.ok) throw new Error('Failed to fetch Bitcoin data');

        const data = await response.json();
        const mainCoinElement = document.getElementById('main-coin');

        const priceChange = data.market_data.price_change_percentage_24h;
        const priceChangeClass = priceChange >= 0 ? 'text-green-400' : 'text-red-500';
        const priceChangeSymbol = priceChange >= 0 ? '▲' : '▼';

        mainCoinElement.innerHTML = `
          <div class="flex flex-col md:flex-row justify-between items-center">
            <div class="flex items-center space-x-4">
              <img src="${data.image.large}" alt="${data.name}" class="w-12 h-12 rounded-full bg-gray-800" />
              <div>
                <h1 class="text-2xl font-bold">
                  ${data.name} <span class="text-gray-500 text-lg">${data.symbol.toUpperCase()}</span>
                </h1>
              </div>
            </div>
            <div class="mt-4 md:mt-0">
              <span class="text-2xl font-bold">${formatCurrency(data.market_data.current_price.usd)}</span>
              <span class="ml-2 font-bold ${priceChangeClass}">
                ${priceChangeSymbol} ${Math.abs(priceChange).toFixed(2)}%
              </span>
            </div>
          </div>
          <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 mt-6">
            <div class="bg-gray-800 p-4 rounded">
              <div class="text-sm text-gray-400">Market Cap</div>
              <div class="text-lg font-bold">${formatLargeNumber(data.market_data.market_cap.usd)}</div>
            </div>
            <div class="bg-gray-800 p-4 rounded">
              <div class="text-sm text-gray-400">24h Trading Volume</div>
              <div class="text-lg font-bold">${formatLargeNumber(data.market_data.total_volume.usd)}</div>
            </div>
            <div class="bg-gray-800 p-4 rounded">
              <div class="text-sm text-gray-400">Circulating Supply</div>
              <div class="text-lg font-bold">${formatNumber(Math.round(data.market_data.circulating_supply))} ${data.symbol.toUpperCase()}</div>
            </div>
            <div class="bg-gray-800 p-4 rounded">
              <div class="text-sm text-gray-400">Total Supply</div>
              <div class="text-lg font-bold">${
                data.market_data.total_supply
                  ? formatNumber(Math.round(data.market_data.total_supply)) + ' ' + data.symbol.toUpperCase()
                  : 'N/A'
              }</div>
            </div>
            <div class="bg-gray-800 p-4 rounded">
              <div class="text-sm text-gray-400">Max Supply</div>
              <div class="text-lg font-bold">${
                data.market_data.max_supply
                  ? formatNumber(Math.round(data.market_data.max_supply)) + ' ' + data.symbol.toUpperCase()
                  : '∞'
              }</div>
            </div>
            <div class="bg-gray-800 p-4 rounded">
              <div class="text-sm text-gray-400">All-Time High</div>
              <div class="text-lg font-bold">${formatCurrency(data.market_data.ath.usd)}</div>
            </div>
          </div>
          <div class="mt-6">
            <div class="w-full h-72 bg-gray-800 rounded flex items-center justify-center">
              <span class="text-gray-400">Chart Placeholder</span>
            </div>
          </div>
        `;
      } catch (error) {
        console.error('Error fetching Bitcoin data:', error);
        document.getElementById('main-coin').innerHTML = `
          <div class="bg-red-100 text-red-500 p-4 rounded text-center">
            <p>Failed to load Bitcoin data. Please try again later.</p>
            <button class="mt-2 bg-green-400 text-black px-4 py-2 rounded" onclick="fetchMainCoinData()">Retry</button>
          </div>
        `;
      }
    }

    // Fetch top coins data
    async function fetchTopCoins() {
      try {
        const response = await fetch('https://api.coingecko.com/api/v3/coins/markets?vs_currency=usd&order=market_cap_desc&per_page=10&page=1');
        if (!response.ok) throw new Error('Failed to fetch top coins');
        const data = await response.json();
        const topCoinsElement = document.getElementById('top-coins');
        topCoinsElement.innerHTML = data
          .map((coin, index) => {
            const priceChange = coin.price_change_percentage_24h;
            const priceChangeClass = priceChange >= 0 ? 'text-green-400' : 'text-red-500';
            const priceChangeSymbol = priceChange >= 0 ? '+' : '';
            return `
              <li class="flex justify-between items-center py-2 border-b border-gray-700">
                <div class="flex items-center space-x-3">
                  <span class="text-gray-400">${index + 1}</span>
                  <img src="${coin.image}" alt="${coin.name}" class="w-6 h-6 rounded-full bg-gray-800" />
                  <span class="font-bold">${coin.name}</span>
                  <span class="text-gray-400">(${coin.symbol.toUpperCase()})</span>
                </div>
                <div class="text-right">
                  <div class="font-bold">${formatCurrency(coin.current_price)}</div>
                  <div class="${priceChangeClass}">${priceChangeSymbol}${priceChange.toFixed(2)}%</div>
                </div>
              </li>
            `;
          })
          .join('');
      } catch (error) {
        console.error('Error fetching top coins:', error);
        document.getElementById('top-coins').innerHTML = `
          <div class="bg-red-100 text-red-500 p-4 rounded text-center">
            <p>Failed to load top coins. Please try again later.</p>
            <button class="mt-2 bg-green-400 text-black px-4 py-2 rounded" onclick="fetchTopCoins()">Retry</button>
          </div>
        `;
      }
    }

    // Fetch recent coins data
    async function fetchRecentCoins() {
      try {
        const response = await fetch('https://api.coingecko.com/api/v3/coins/markets?vs_currency=usd&order=id_desc&per_page=10&page=1');
        if (!response.ok) throw new Error('Failed to fetch recent coins');
        const data = await response.json();
        const recentCoinsElement = document.getElementById('recent-coins');
        recentCoinsElement.innerHTML = data
          .map((coin) => {
            const priceChange = coin.price_change_percentage_24h;
            const priceChangeClass = priceChange >= 0 ? 'text-green-400' : 'text-red-500';
            const priceChangeSymbol = priceChange >= 0 ? '+' : '';
            return `
              <li class="flex justify-between items-center py-2 border-b border-gray-700">
                <div class="flex items-center space-x-3">
                  <img src="${coin.image}" alt="${coin.name}" class="w-6 h-6 rounded-full bg-gray-800" />
                  <span class="font-bold">${coin.name}</span>
                  <span class="text-gray-400">(${coin.symbol.toUpperCase()})</span>
                </div>
                <div class="text-right">
                  <div class="font-bold">${formatCurrency(coin.current_price)}</div>
                  <div class="${priceChangeClass}">${priceChangeSymbol}${priceChange.toFixed(2)}%</div>
                </div>
              </li>
            `;
          })
          .join('');
      } catch (error) {
        console.error('Error fetching recent coins:', error);
        document.getElementById('recent-coins').innerHTML = `
          <div class="bg-red-100 text-red-500 p-4 rounded text-center">
            <p>Failed to load recent coins. Please try again later.</p>
            <button class="mt-2 bg-green-400 text-black px-4 py-2 rounded" onclick="fetchRecentCoins()">Retry</button>
          </div>
        `;
      }
    }

    // Load all data when the DOM content is loaded
    document.addEventListener('DOMContentLoaded', () => {
      fetchMainCoinData();
      fetchTopCoins();
      fetchRecentCoins();
    });
  </script>
</body>
</html>
